/*:
 * @target MZ
 * @plugindesc Global Slow Zoom v1.5 – イージング + 完了待ち + 中心も滑らか補間でカクつきゼロ
 * @author YourName
 * -------------------------------------------------------------------
 * @command zoom
 * @text ズーム開始
 *
 * @arg scale     @type number  @decimals 2 @min 0.10 @default 1.20
 * @arg duration  @type number  @min 1      @default 60
 * @arg centerX   @type number  @default 0  @text 中心X(px) 0=画面中央
 * @arg centerY   @type number  @default 0  @text 中心Y(px)
 * @arg easing    @type select  @option linear @option easeInOut @default linear
 * @arg wait      @type boolean @on 待つ @off 待たない @default false
 * -------------------------------------------------------------------
 * @command reset
 * @text ズームリセット
 *
 * @arg duration  @type number  @min 1      @default 60
 * @arg easing    @type select  @option linear @option easeInOut @default linear
 * @arg wait      @type boolean @on 待つ @off 待たない @default false
 * -------------------------------------------------------------------
 * @help
 * ■v1.5 追加点
 *   1. "中心座標" もズーム時間と同じイージングで補間 → 連続ズーム時に Y を変えても一切ジャンプしません。
 *   2. コマンド引数・使い方は v1.4.2 と同じ（置き換えるだけで動きます）。
 *
 * ■イージング種類
 *   linear      : 一定速度
 *   easeInOut   : ゆっくり加速→ゆっくり減速（Cosine 補間）
 *
 * ■wait=true
 *   ズーム完了までイベントコマンドを停止します。
 */

(() => {
  const PLUGIN = "GlobalSlowZoom";
  const num = (v, def) => (v === undefined ? def : Number(v));

  // ---------------- ZoomManager ----------------
  const ZoomManager = {
    init() {
      this._startScale = this._targetScale = 1;
      this._startCx = this._targetCx = Graphics.width  / 2;
      this._startCy = this._targetCy = Graphics.height / 2;
      this._dur = this._elapsed = 0;
      this._easing = "linear";
      this._waiters = [];
    },

    start(targetScale, dur, cx, cy, easing, interpreter) {
      this._startScale  = this.currentScale();
      this._targetScale = targetScale;
      this._dur         = Math.max(1, dur);
      this._elapsed     = 0;
      this._startCx     = this._currentCx();
      this._startCy     = this._currentCy();
      this._targetCx    = cx !== undefined ? (cx || Graphics.width  / 2) : this._startCx;
      this._targetCy    = cy !== undefined ? (cy || Graphics.height / 2) : this._startCy;
      this._easing      = easing || "linear";
      if (interpreter) this._waiters.push(interpreter);
    },

    currentScale() { return SceneManager._scene ? SceneManager._scene.scale.x : 1; },
    _currentCx()   { return this._lastCx ?? Graphics.width  / 2; },
    _currentCy()   { return this._lastCy ?? Graphics.height / 2; },
    isBusy()       { return this._elapsed < this._dur; },

    update() {
      if (!SceneManager._scene || !this.isBusy()) return;

      this._elapsed++;
      const t   = this._elapsed / this._dur;
      const r   = this._easing === "easeInOut" ? (1 - Math.cos(Math.PI * t)) / 2 : t;
      const sc  = this._startScale + (this._targetScale - this._startScale) * r;
      const cx  = this._startCx   + (this._targetCx    - this._startCx)   * r;
      const cy  = this._startCy   + (this._targetCy    - this._startCy)   * r;
      this.apply(sc, cx, cy);

      if (!this.isBusy()) this.finishWaiters();
    },

    apply(scale, cx, cy) {
      const scn = SceneManager._scene;
      scn.scale.set(scale, scale);
      const offX = (Graphics.width  / 2 - cx) * (scale - 1);
      const offY = (Graphics.height / 2 - cy) * (scale - 1);
      scn.position.set(offX, offY);
      this._lastCx = cx; // 次回の start() で現在値にできるよう保持
      this._lastCy = cy;
    },

    finishWaiters() {
      while (this._waiters.length) {
        const itp = this._waiters.shift();
        if (itp._waitMode === "zoom") itp.setWaitMode("");
      }
    }
  };
  ZoomManager.init();

  // ---------------- Plugin Commands ------------
  PluginManager.registerCommand(PLUGIN, "zoom", function (args) {
    const scale  = num(args.scale, 1);
    const dur    = num(args.duration, 60);
    const cx     = num(args.centerX, 0);
    const cy     = num(args.centerY, 0);
    const easing = args.easing || "linear";
    const wait   = args.wait === "true";
    if (wait) this.setWaitMode("zoom");
    ZoomManager.start(scale, dur, cx, cy, easing, wait ? this : null);
  });

  PluginManager.registerCommand(PLUGIN, "reset", function (args) {
    const dur    = num(args.duration, 60);
    const easing = args.easing || "linear";
    const wait   = args.wait === "true";
    if (wait) this.setWaitMode("zoom");
    ZoomManager.start(1, dur, undefined, undefined, easing, wait ? this : null);
  });

  // ---------------- Scene Hook -------------
  const _SceneManager_updateScene = SceneManager.updateScene;
  SceneManager.updateScene = function () {
    _SceneManager_updateScene.call(this);
    ZoomManager.update();
  };

  // ---------------- Interpreter Wait -------
  const _GI_updateWait = Game_Interpreter.prototype.updateWait;
  Game_Interpreter.prototype.updateWait = function () {
    if (this._waitMode === "zoom") {
      if (ZoomManager.isBusy()) return true;
      this._waitMode = "";
    }
    return _GI_updateWait.call(this);
  };

  // ---------------- TouchInput Scaling -----
  const _TouchInput_update = TouchInput.update;
  TouchInput.update = function () {
    _TouchInput_update.call(this);
    const scn = SceneManager._scene;
    if (scn && scn.scale.x) {
      this._scaledX = (this.x - scn.x) / scn.scale.x;
      this._scaledY = (this.y - scn.y) / scn.scale.y;
    } else {
      this._scaledX = this.x;
      this._scaledY = this.y;
    }
  };
  Object.defineProperty(TouchInput, "scaledX", { get() { return this._scaledX; } });
  Object.defineProperty(TouchInput, "scaledY", { get() { return this._scaledY; } });
})();
